/**
* \file: WaylandFacade.cpp
*
* \version: 0.4
*
* \release: $Name:$
*
* Main interaction point with the individual SPI components, simplifies
* the use of Wayland
*
* \component: Unified SPI
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <string>
#include "WaylandConnection.h"
#include <uspi/WaylandFacade.h>

namespace adit { namespace uspi {
using std::unique_ptr;
using std::move;
using std::to_string;

WaylandFacade::WaylandFacade(void* inSessionCtx)
: ITouchFacade(inSessionCtx), SharedDataReceiver(inSessionCtx), mCallbacks(nullptr), mWlConnection(nullptr),
  mWlDisplay(nullptr), mDisplayWidth(0), mDisplayHeight(0), mMaxFingerNum(1), mConnected(false) { }

WaylandFacade::~WaylandFacade()
{
    if (!deinitialize())
    {
        mCallbacks->onLogging(USPI_LOG_ERROR, "Destruction cannot deinitialize Wayland Facade");
    }
}

WaylandFacade* WaylandFacade::setDisplaySize(int inDisplayWidth, int inDisplayHeight)
{
    mDisplayWidth = inDisplayWidth;
    mDisplayHeight = inDisplayHeight;

    return this;
}

WaylandFacade* WaylandFacade::setFingerNum(int inMaxFingerNum)
{
    mMaxFingerNum = inMaxFingerNum;

    return this;
}

bool WaylandFacade::initialize(ITouchFacadeCallbacks* inCallbacks)
{
    uspi_return_value_on_invalid_argument(uspi, inCallbacks == nullptr, false);

    mCallbacks = inCallbacks;

    int ret_val = SharedDataSender::instance().subscribeForDataSharing(*this);
    if (ret_val != 0)
    {
        mCallbacks->onLogging(USPI_LOG_ERROR, "WaylandFacade cannot subscribe for notifications, err: "
                + to_string(ret_val));
        return false;
    }

    mWlConnection = move(unique_ptr<WaylandConnection>(new WaylandConnection(*mCallbacks, mDisplayWidth,
            mDisplayHeight, mMaxFingerNum)));

    return true;
}

bool WaylandFacade::deinitialize()
{
    if (!disconnect())
    {
        mCallbacks->onLogging(USPI_LOG_ERROR, "WaylandFacade cannot disconnect from Wayland");
        return false;
    }

    int ret_val = SharedDataSender::instance().unsubscribeFromDataSharing(*this);
    if (ret_val == -1) {
        mCallbacks->onLogging(USPI_LOG_DEBUG, "SessionContext was already unsubscribed");
    } else if (ret_val != 0) {
        mCallbacks->onLogging(USPI_LOG_ERROR, "WaylandFacade cannot unsubscribe, err: " + to_string(ret_val));
        return false;
    }
    return true;
}

bool WaylandFacade::connect()
{
    if (mConnected)
    {
        mCallbacks->onLogging(USPI_LOG_DEBUG, "Wayland Facade is already connected");
        return true;
    }

    if (!mWlConnection->start(mWlDisplay))
    {
        mCallbacks->onLogging(USPI_LOG_ERROR, "Wayland Facade could not start Wayland Connection");
        return false;
    }

    mConnected = true;
    return true;
}

bool WaylandFacade::disconnect()
{
    if (!mConnected)
    {
        mCallbacks->onLogging(USPI_LOG_DEBUG, "Wayland Facade is already disconnected");
        return true;
    }

    if (!mWlConnection->stop())
    {
        mCallbacks->onLogging(USPI_LOG_ERROR, "Wayland Facade could not stop Wayland Connection");
        return false;
    }

    mConnected = false;
    return true;
}

void WaylandFacade::onSurfaceDataProvided(wl_display* inWlDisplay)
{
    uspi_return_on_invalid_argument(uspi, inWlDisplay == nullptr);

    mWlDisplay = inWlDisplay;
    if (!connect())
    {
        mCallbacks->onLogging(USPI_LOG_ERROR, "Starting Wayland Display Provider could not initiate Wayland Facade to connect");
    }
}

void WaylandFacade::onSurfaceDataExpired()
{
    if (!disconnect())
    {
        mCallbacks->onLogging(USPI_LOG_ERROR, "Stopping Wayland Display Provider could not initiate Wayland Facade to disconnect");
    }
}

void WaylandFacade::onResolutionDataProvided(int inWidth, int inHeight, int inMarginX, int inMarginY)
{
    mDisplayWidth = inWidth;
    mDisplayHeight = inHeight;

    mWlConnection->setLateResolution(mDisplayWidth, mDisplayHeight);

    mCallbacks->onLateResolutionData(inWidth, inHeight, inMarginX, inMarginY);
}

} } /* namespace adit { namespace uspi { */
